﻿using log4net;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Formatting;
using System.Net.Http.Headers;
using System.ServiceModel;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.BatchService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.BatchService
{
    public class BatchService : IBatchService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IPpmsContextHelper _ppmsContextHelper;
        private readonly IPpmsHelper _ppmsHelper;


        public BatchService(ILog logger, IIwsConfiguration configuration, IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
        {
            _logger = logger;
            _configuration = configuration;
            _ppmsContextHelper = ppmsContextHelper;
            _ppmsHelper = ppmsHelper;
        }

        public async Task ProcessBatch(string queueMessage)
        {
            try
            {
                if (string.IsNullOrEmpty(queueMessage))
                {
                    _logger.Info($"@@@@ INFO - Parameter invalid for item: {queueMessage} @@@@");
                    return;
                }

                var message = new DasMessage(queueMessage);
                var batchId = message.Content;
                var batch = await GetBatch(batchId);

                _logger.Info($"@@@@ INFO - Verify details for BatchId: {batchId} @@@@");
                var details = batch.ppms_batch_batchdetail_batch;
                IEnumerable<ppms_batchdetail> ppmsBatchdetails = details as ppms_batchdetail[] ?? details.ToArray();
                if (!ppmsBatchdetails.Any())
                {
                    _logger.Info($"@@@@ INFO - Details not found for BatchId: {batchId} @@@@");
                    return;
                }

                _logger.Info($"@@@@ INFO - Details found for BatchId: {batchId} @@@@");

                // Create XML doc
                var doc = new ProviderResponses
                {
                    ProviderResponse = new List<ProviderResponse>(),
                    TransactionId = batch.ppms_transactionid
                };

                // Capture batch details
                foreach (var detail in ppmsBatchdetails)
                {
                    // Provider node
                    var provider = new ProviderResponse
                    {
                        ProviderId = detail.ppms_providerid,
                        Success = detail.GetAttributeValue<bool>("ppms_isvalid")
                    };

                    // Set correlation id, if appropriate
                    if (detail.ppms_provider != null)
                    {
                        provider.CorrelationId = detail.ppms_provider.Id.ToString();
                    }

                    // Retrieve batch detail results
                    var batchDetailResults = detail.ppms_batchdetail_batchdetailresult;

                    // Capture batch detail results
                    IEnumerable<ppms_batchdetailresult> ppmsBatchdetailresults = batchDetailResults as ppms_batchdetailresult[] ?? batchDetailResults.ToArray();
                    if (ppmsBatchdetailresults.Any())
                    {
                        // Initialize results list
                        provider.Results = new Results { Item = new List<Result>() };

                        foreach (var detailResult in ppmsBatchdetailresults)
                        {
                            var result = new Result
                            {
                                Type = detailResult.ppms_entitytype,
                                Id = detailResult.ppms_name,
                                Success = detailResult.ppms_isvalid.HasValue && detailResult.ppms_isvalid.Value
                            };

                            if (!result.Success)
                            {
                                result.Header = detailResult.ppms_result;
                                result.Message = detailResult.ppms_message;
                            }

                            provider.Results.Item.Add(result);
                        }
                    }
                    else
                    {
                        _logger.Info($"@@@@ INFO - Results not found for provider: {detail.ppms_name} @@@@");
                    }

                    // Add to list of providers
                    doc.ProviderResponse.Add(provider);
                }

                // Create response packet
                var responseToSend = await ConvertResponseToXml(doc);

                // Send response
                await PostXmlToDas(message, responseToSend);

                // Close batch
                await _ppmsHelper.UpdateBatch(message, "CCN Response sent.", (int)ppms_batch_StatusCode.Complete);

                _logger.Info($"@@@@ INFO - Save complete for BatchId: {batchId} @@@@");
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                _logger.Error($"@@@@ CrResponseQueueTrigger ERROR - Fault: {ex} @@@@", ex);
                throw;
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CrResponseQueueTrigger ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        public async Task<bool> CreateBatch(DasMessage dasMessage, string updateMessage, int statusCode)
        {
            try
            {
                return await _ppmsHelper.CreateBatch(dasMessage, updateMessage, statusCode);
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CreateBatch ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        public async Task<bool> UpdateBatch(DasMessage dasMessage, string updateMessage, int statusCode)
        {
            try
            {
                return await _ppmsHelper.UpdateBatch(dasMessage, updateMessage, statusCode);
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CreateBatch ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        public async Task IncrementBatchStatus(DasMessage message, string activityMessage)
        {
            try
            {
                await _ppmsHelper.IncrementBatchStatus(message, activityMessage);
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ CreateBatch ERROR - Exception: {ex} @@@@", ex);
                throw;
            }
        }

        private async Task<ppms_batch> GetBatch(string batchId)
        {
            var context = await _ppmsContextHelper.GetContextAsync();
            var batch = context.ppms_batchSet.FirstOrDefault(b => b.ppms_batchId == new Guid(batchId));
            if (batch == null) throw new PpmsServiceException("Batch record does not exist");

            context.LoadProperty(batch, new Relationship("ppms_batch_batchdetail_batch"));

            if (batch.ppms_batch_batchdetail_batch == null) return batch;
            foreach (var item in batch.ppms_batch_batchdetail_batch)
            {
                context.LoadProperty(item, new Relationship("ppms_batchdetail_batchdetailresult"));
            }

            return batch;
        }

        private async Task<string> ConvertResponseToXml(ProviderResponses response)
        {
            var packet = await _configuration.GetSchemaProfileAsync(SchemaOptions.SchemaProfiles.Response);

            var prefix = packet.Prefix;
            var nameSpace = packet.Namespace;

            return Utilities.SerializeInstance(response, prefix, nameSpace);
        }

        private async Task PostXmlToDas(DasMessage message, string content)
        {
            var baseUri = await _configuration.GetPpmsResponseNotificationUriAsync();
            var requestUri = await _configuration.GetPpmsResponsePostUriAsync();

            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(baseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/xml"));
                client.DefaultRequestHeaders.Add("X-ConversationID", message.ConversationId);
                client.DefaultRequestHeaders.Add("X-RoutingSenderID", message.SenderId);
                client.DefaultRequestHeaders.Add("X-RoutingReceiverIDs", message.ReceiverId);
                client.DefaultRequestHeaders.Add("X-TransactionID", message.TransactionId);

                // TODO - This should be tested in isolation against DAS' endpoint to determine what does and does not work.
                await client.PostAsync(requestUri, content, new XmlMediaTypeFormatter { UseXmlSerializer = true });
            }
        }
    }
}